package nemosofts.notes.app.activity;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.DatePickerDialog;
import android.app.TimePickerDialog;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.GradientDrawable;
import android.media.MediaPlayer;
import android.media.MediaRecorder;
import android.media.MediaScannerConnection;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.os.Handler;
import android.os.Looper;
import android.text.Html;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import androidx.activity.EdgeToEdge;
import androidx.activity.OnBackPressedCallback;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.widget.Toolbar;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.core.graphics.Insets;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.nemosofts.AppCompatActivity;
import androidx.nemosofts.material.ImageViewRound;
import androidx.nemosofts.material.Toasty;
import androidx.nemosofts.theme.ColorUtils;
import androidx.nemosofts.theme.ThemeEngine;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.google.android.material.bottomsheet.BottomSheetBehavior;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Random;

import nemosofts.notes.app.R;
import nemosofts.notes.app.adapters.AdapterCategories;
import nemosofts.notes.app.callback.Callback;
import nemosofts.notes.app.dialog.DialogUtil;
import nemosofts.notes.app.item.Cat;
import nemosofts.notes.app.item.Note;
import nemosofts.notes.app.item.NoteHistory;
import nemosofts.notes.app.listeners.ClickListenerPlayList;
import nemosofts.notes.app.utils.AsyncTaskExecutor;
import nemosofts.notes.app.utils.IfSupported;
import nemosofts.notes.app.utils.MediaPath;
import nemosofts.notes.app.utils.ReminderScheduler;
import nemosofts.notes.app.utils.helper.DBHelper;
import nemosofts.notes.app.utils.helper.Helper;
import nemosofts.notes.app.view.NemosoftsEditText;

public class CreateNoteActivity extends AppCompatActivity {

    private static final String TAG = "CreateNoteActivity";
    private static final String COLOR_DEFAULT = "#333333";
    private static final String COLOR_AMBER = "#fdbe3b";
    private static final String COLOR_PURPLE = "#E040FB";
    private static final String COLOR_BLUE = "#3a52fc";
    private static final String COLOR_PINK = "#F50057";
    private static final String COLOR_GREEN = "#00E676";
    private static final String COLOR_ORANGE = "#FF3D00";

    private static final String ERROR_TITLE_EMPTY = "Note title can't be empty!";
    private static final String ERROR_SUBTITLE_EMPTY = "Note subtitle can't be empty!";
    private static final String ERROR_NOTE_EMPTY = "Note can't be empty!";

    private static final int PERMISSION_REQ_WRITE_EXTERNAL_STORAGE = 102;
    private static final int REQ_POST_NOTIFICATIONS = 301;
    private static final int REQ_RECORD_AUDIO = 401;

    private EditText inpuNoteTitle;
    private EditText inpuNoteSubtitle;
    private TextView textDeteTime;
    private TextView textMeta;
    private TextView textWebURL;
    private TextView textReminder;
    private ImageViewRound imageNote;
    private String setectedNoteColor;
    private View viewSubtitleIndicator;
    private LinearLayout layoutWebURL;
    private LinearLayout layoutReminder;
    private LinearLayout layoutAudio;
    private TextView textAudioDuration;
    private ImageView buttonAudioAction;
    private ImageView buttonAudioRemove;
    private long reminderTime = 0L;
    private String setectedImagePath;
    private String audioPath = "";
    private long audioDuration = 0L;
    private AlertDialog dialogExport;
    private final ActivityResultLauncher<String> imagePickerLauncher =
            registerForActivityResult(new ActivityResultContracts.GetContent(), this::handleImagePicked);

    private Note alreadyAvailableNote;
    private NemosoftsEditText nemosoftsEditText;
    private int generator;
    private LinearLayout llScroll;

    private LinearLayout linearMiscellaneous;
    private BottomSheetBehavior<LinearLayout> bottomSheetBehavior;

    private DBHelper dbHelper;
    private AdapterCategories adapterCategories;
    private ArrayList<Cat> arrayList;
    private String cid = "1";

    private static final Random RANDOM = new Random();
    private MediaRecorder mediaRecorder;
    private MediaPlayer mediaPlayer;
    private boolean isRecording = false;
    private boolean isPlaying = false;
    private long recordStartTime = 0L;
    private final Handler recordingHandler = new Handler(Looper.getMainLooper());
    private Runnable recordingTicker;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        IfSupported.isRTL(this);
        IfSupported.isScreenshot(this);
        EdgeToEdge.enable(this);
        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.bg_page), (v, insets) -> {
            Insets systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars());
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom);
            return insets;
        });

        Toolbar toolbar = findViewById(R.id.toolbar);
        setSupportActionBar(toolbar);
        if (getSupportActionBar() != null) {
            getSupportActionBar().setDisplayHomeAsUpEnabled(true);
            getSupportActionBar().setDisplayShowHomeEnabled(true);
        }
        toolbar.setNavigationOnClickListener(view -> onBack());

        llScroll = findViewById(R.id.llScroll);
        linearMiscellaneous = findViewById(R.id.layoutMiscellaneous);
        bottomSheetBehavior = BottomSheetBehavior.from(linearMiscellaneous);
        inpuNoteTitle = findViewById(R.id.create_inputNoteTitle);
        inpuNoteSubtitle = findViewById(R.id.create_inputNoteSubtitle);
        textDeteTime = findViewById(R.id.create_textDeteTime);
        textMeta = new TextView(this);
        textMeta.setTextColor(ColorUtils.colorTitle(this));
        textMeta.setTextSize(12f);
        llScroll.addView(textMeta, 1);
        viewSubtitleIndicator = findViewById(R.id.create_viewSubtitleIndicator);
        imageNote = findViewById(R.id.create_imageNote);
        textWebURL = findViewById(R.id.create_textWebURL);
        layoutWebURL = findViewById(R.id.create_layoutWebURL);
        textReminder = findViewById(R.id.create_textReminder);
        layoutReminder = findViewById(R.id.create_layoutReminder);
        layoutAudio = findViewById(R.id.create_layoutAudio);
        textAudioDuration = findViewById(R.id.create_textAudioDuration);
        buttonAudioAction = findViewById(R.id.create_buttonAudioAction);
        buttonAudioRemove = findViewById(R.id.create_imageRemoveAudio);
        nemosoftsEditText = findViewById(R.id.create_inputNote);

        nemosoftsEditText.setSelection(nemosoftsEditText.getEditableText().length());

        textDeteTime.setText(
                new SimpleDateFormat("EEEE , dd MMMM yyyy HH:mm a", Locale.getDefault())
                .format(new Date())
        );
        textMeta.setText(getString(R.string.last_updated_now));

        setectedNoteColor = COLOR_DEFAULT;
        setectedImagePath = "";

        dbHelper = new DBHelper(this);
        arrayList = new ArrayList<>();
        arrayList.addAll(dbHelper.loadPlayList());

        if (getIntent().getBooleanExtra("isViewOrUpdate",false)){
            alreadyAvailableNote = (Note) getIntent().getSerializableExtra("note");
            setViewOrUpdateNote();
            touchLastOpened();
        } else {
            findViewById(R.id.iv_delete).setVisibility(View.GONE);
        }

        initMiscellaneous();
        setSubtitleIndicatorColor();
        setClickListener();
        setCategories();

        OnBackPressedCallback callback = new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                onBack();
            }
        };
        getOnBackPressedDispatcher().addCallback(this, callback);

        LinearLayout adView = findViewById(R.id.ll_adView);
        new Helper(this).showBannerAd(adView);

        findViewById(R.id.iv_save).setOnClickListener(v -> saveNote());
        findViewById(R.id.iv_delete).setOnClickListener(v -> deleteNote());
    }

    private void setCategories() {
        RecyclerView rv = findViewById(R.id.rv_cat);
        LinearLayoutManager manager = new LinearLayoutManager(this, LinearLayoutManager.HORIZONTAL, false);
        rv.setLayoutManager(manager);
        rv.setItemAnimator(new DefaultItemAnimator());
        adapterCategories = new AdapterCategories(this, arrayList, new ClickListenerPlayList() {
            @Override
            public void onClick(int position) {
                adapterCategories.select(Integer.parseInt(arrayList.get(position).id()));
                cid = arrayList.get(position).id();
            }

            @SuppressLint("NotifyDataSetChanged")
            @Override
            public void onItemZero() {
                arrayList.clear();
                arrayList.addAll(dbHelper.loadPlayList());
                adapterCategories.notifyDataSetChanged();
            }
        });
        rv.setAdapter(adapterCategories);
        adapterCategories.select(Integer.parseInt(cid));
    }

    private void deleteNote() {
        if (alreadyAvailableNote == null){
            return;
        }
        showDeletNoteDialog();
    }

    private void onBack() {
        if (bottomSheetBehavior!=null && bottomSheetBehavior.getState() != BottomSheetBehavior.STATE_COLLAPSED) {
            bottomSheetBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
        } else if (alreadyAvailableNote == null){
            if (inpuNoteTitle.getText().toString().trim().isEmpty()
                    || inpuNoteSubtitle.getText().toString().trim().isEmpty()
                    || nemosoftsEditText.getText().toString().trim().isEmpty()){
                finish();
            } else {
                saveNoteOnBackPressed();
            }
        } else {
            finish();
        }
    }

    private void setClickListener() {
        findViewById(R.id.button_add).setOnClickListener(view -> openAddListDialog());
        textMeta.setOnClickListener(v -> showHistoryDialog());
        findViewById(R.id.create_imageRemoveWebURL).setOnClickListener(view -> {
            textWebURL.setText(null);
            layoutWebURL.setVisibility(View.GONE);
            if (alreadyAvailableNote != null){
                alreadyAvailableNote.setWebLink("");
            }
        });
        findViewById(R.id.create_imageRemoveImage).setOnClickListener(view -> {
            imageNote.setImageBitmap(null);
            imageNote.setVisibility(View.GONE);
            findViewById(R.id.create_imageRemoveImage).setVisibility(View.GONE);
            if (alreadyAvailableNote != null){
                alreadyAvailableNote.setImagePath("");
            }
            setectedImagePath = "";
        });
        findViewById(R.id.create_imageRemoveReminder).setOnClickListener(view -> clearReminder());
        buttonAudioRemove.setOnClickListener(v -> clearAudio());
        buttonAudioAction.setOnClickListener(v -> togglePlay());

        findViewById(R.id.bold).setOnClickListener(v -> nemosoftsEditText.bold(!nemosoftsEditText.contains(NemosoftsEditText.FORMAT_BOLD)));
        findViewById(R.id.bold).setOnLongClickListener(v -> {
            Toast.makeText(CreateNoteActivity.this, R.string.toast_bold, Toast.LENGTH_SHORT).show();
            return true;
        });

        findViewById(R.id.italic).setOnClickListener(v -> nemosoftsEditText.italic(!nemosoftsEditText.contains(NemosoftsEditText.FORMAT_ITALIC)));
        findViewById(R.id.italic).setOnLongClickListener(v -> {
            Toast.makeText(CreateNoteActivity.this, R.string.toast_italic, Toast.LENGTH_SHORT).show();
            return true;
        });

        findViewById(R.id.underline).setOnClickListener(v -> nemosoftsEditText.underline(!nemosoftsEditText.contains(NemosoftsEditText.FORMAT_UNDERLINED)));
        findViewById(R.id.underline).setOnLongClickListener(v -> {
            Toast.makeText(CreateNoteActivity.this, R.string.toast_underline, Toast.LENGTH_SHORT).show();
            return true;
        });

        findViewById(R.id.strikethrough).setOnClickListener(v -> nemosoftsEditText.strikethrough(!nemosoftsEditText.contains(NemosoftsEditText.FORMAT_STRIKETHROUGH)));
        findViewById(R.id.strikethrough).setOnLongClickListener(v -> {
            Toast.makeText(CreateNoteActivity.this, R.string.toast_strikethrough, Toast.LENGTH_SHORT).show();
            return true;
        });

        findViewById(R.id.bullet).setOnClickListener(v -> nemosoftsEditText.bullet(!nemosoftsEditText.contains(NemosoftsEditText.FORMAT_BULLET)));
        findViewById(R.id.bullet).setOnLongClickListener(v -> {
            Toast.makeText(CreateNoteActivity.this, R.string.toast_bullet, Toast.LENGTH_SHORT).show();
            return true;
        });

        findViewById(R.id.quote).setOnClickListener(v -> nemosoftsEditText.quote(!nemosoftsEditText.contains(NemosoftsEditText.FORMAT_QUOTE)));
        findViewById(R.id.quote).setOnLongClickListener(v -> {
            Toast.makeText(CreateNoteActivity.this, R.string.toast_quote, Toast.LENGTH_SHORT).show();
            return true;
        });

        findViewById(R.id.clear).setOnClickListener(v -> nemosoftsEditText.clearFormats());
        findViewById(R.id.clear).setOnLongClickListener(v -> {
            Toast.makeText(CreateNoteActivity.this, R.string.toast_format_clear, Toast.LENGTH_SHORT).show();
            return true;
        });
    }

    @Override
    public int setContentViewID() {
        return R.layout.activity_create_note;
    }

    @SuppressLint("NotifyDataSetChanged")
    private void openAddListDialog() {
        DialogUtil.addCategoriesDialog(this, string -> {
            arrayList.clear();
            arrayList.addAll(dbHelper.addPlayList(string));
            Toast.makeText(CreateNoteActivity.this, getString(R.string.add_categories), Toast.LENGTH_SHORT).show();
            adapterCategories.notifyDataSetChanged();
        });
    }

    private void setViewOrUpdateNote() {
        inpuNoteTitle.setText(alreadyAvailableNote.getTitle());
        inpuNoteSubtitle.setText(alreadyAvailableNote.getSubtitle());
        nemosoftsEditText.fromHtml(alreadyAvailableNote.getNoteText());
        textDeteTime.setText(alreadyAvailableNote.getDateTime());
        cid = alreadyAvailableNote.getCatId();

        if (alreadyAvailableNote.getImagePath() != null && !alreadyAvailableNote.getImagePath().trim().isEmpty()){
            setectedImagePath = alreadyAvailableNote.getImagePath();
            loadNoteImage(setectedImagePath);
        }

        if (alreadyAvailableNote.getWebLink() != null && !alreadyAvailableNote.getWebLink().trim().isEmpty()){
            textWebURL.setText(alreadyAvailableNote.getWebLink());
            layoutWebURL.setVisibility(View.VISIBLE);
        }

        if (alreadyAvailableNote.getAudioPath() != null && !alreadyAvailableNote.getAudioPath().trim().isEmpty()) {
            audioPath = alreadyAvailableNote.getAudioPath();
            audioDuration = alreadyAvailableNote.getAudioDuration();
            showAudio(audioDuration);
        }

        updateMetaInfo();

        reminderTime = alreadyAvailableNote.getReminderTime();
        if (alreadyAvailableNote.isReminderEnabled() && reminderTime > 0){
            updateReminderUI();
        } else {
            clearReminder();
        }
    }

    private void updateMetaInfo() {
        String created = alreadyAvailableNote != null ? alreadyAvailableNote.getDateTime() : textDeteTime.getText().toString();
        StringBuilder builder = new StringBuilder(getString(R.string.created_at, created));
        long updatedTs = alreadyAvailableNote != null ? alreadyAvailableNote.getLastUpdated() : 0L;
        long openedTs = alreadyAvailableNote != null ? alreadyAvailableNote.getLastOpened() : 0L;
        if (updatedTs > 0) {
            builder.append("\n").append(getString(R.string.last_updated_at, formatTimestamp(updatedTs)));
        }
        if (openedTs > 0) {
            builder.append("\n").append(getString(R.string.last_opened_at, formatTimestamp(openedTs)));
        }
        textMeta.setText(builder.toString());
    }

    private String formatTimestamp(long timestamp) {
        return new SimpleDateFormat("dd MMM yyyy, HH:mm", Locale.getDefault()).format(new Date(timestamp));
    }

    private void touchLastOpened() {
        if (alreadyAvailableNote == null || alreadyAvailableNote.getId() <= 0) {
            return;
        }
        long now = System.currentTimeMillis();
        alreadyAvailableNote.setLastOpened(now);
        dbHelper.updateLastOpened(alreadyAvailableNote.getId(), now);
        dbHelper.addHistory(alreadyAvailableNote.getId(), "Opened", alreadyAvailableNote.getTitle(), now);
        updateMetaInfo();
    }

    private void showHistoryDialog() {
        if (alreadyAvailableNote == null || alreadyAvailableNote.getId() <= 0) {
            Toast.makeText(this, R.string.no_history, Toast.LENGTH_SHORT).show();
            return;
        }
        List<NoteHistory> history = dbHelper.getHistory(alreadyAvailableNote.getId());
        if (history.isEmpty()) {
            Toast.makeText(this, R.string.no_history, Toast.LENGTH_SHORT).show();
            return;
        }
        StringBuilder builder = new StringBuilder();
        for (NoteHistory item : history) {
            builder.append(formatTimestamp(item.getTimestamp()))
                    .append(" • ")
                    .append(item.getAction());
            if (item.getDetail() != null && !item.getDetail().isEmpty()) {
                builder.append(" (").append(item.getDetail()).append(")");
            }
            builder.append("\n");
        }
        new AlertDialog.Builder(this)
                .setTitle(R.string.activity_log_title)
                .setMessage(builder.toString().trim())
                .setPositiveButton(android.R.string.ok, null)
                .show();
    }

    private void openReminderPicker() {
        Calendar now = Calendar.getInstance();
        DatePickerDialog datePickerDialog = new DatePickerDialog(
                this,
                (view, year, month, dayOfMonth) -> {
                    Calendar calendar = Calendar.getInstance();
                    calendar.set(Calendar.YEAR, year);
                    calendar.set(Calendar.MONTH, month);
                    calendar.set(Calendar.DAY_OF_MONTH, dayOfMonth);

                    TimePickerDialog timePickerDialog = new TimePickerDialog(
                            this,
                            (view1, hourOfDay, minute) -> {
                                calendar.set(Calendar.HOUR_OF_DAY, hourOfDay);
                                calendar.set(Calendar.MINUTE, minute);
                                calendar.set(Calendar.SECOND, 0);
                                calendar.set(Calendar.MILLISECOND, 0);

                                if (calendar.getTimeInMillis() <= System.currentTimeMillis()) {
                                    Toasty.makeText(this, getString(R.string.reminder_time_invalid), Toasty.ERROR);
                                    return;
                                }
                                reminderTime = calendar.getTimeInMillis();
                                updateReminderUI();
                            },
                            now.get(Calendar.HOUR_OF_DAY),
                            now.get(Calendar.MINUTE),
                            android.text.format.DateFormat.is24HourFormat(this)
                    );
                    timePickerDialog.show();
                },
                now.get(Calendar.YEAR),
                now.get(Calendar.MONTH),
                now.get(Calendar.DAY_OF_MONTH)
        );
        datePickerDialog.show();
    }

    private Note buildNoteForSave() {
        Note note = new Note();
        note.setTitle(inpuNoteTitle.getText().toString());
        note.setSubtitle(inpuNoteSubtitle.getText().toString());
        note.setNoteText(nemosoftsEditText.toHtml());
        note.setDateTime(textDeteTime.getText().toString());
        note.setColor(setectedNoteColor);
        if (alreadyAvailableNote != null && (setectedImagePath == null || setectedImagePath.trim().isEmpty())) {
            setectedImagePath = alreadyAvailableNote.getImagePath();
        }
        note.setImagePath(setectedImagePath);
        note.setCatId(cid);

        if (layoutWebURL.getVisibility() == View.VISIBLE){
            note.setWebLink(textWebURL.getText().toString());
        }

        note.setReminderTime(reminderTime);
        note.setReminderEnabled(reminderTime > 0);
        note.setAudioPath(audioPath);
        note.setAudioDuration(audioDuration);
        note.setLastUpdated(System.currentTimeMillis());

        if (alreadyAvailableNote != null){
            note.setId(alreadyAvailableNote.getId());
            note.setLastOpened(alreadyAvailableNote.getLastOpened());
        }
        return note;
    }

    private void saveNote() {
        if (inpuNoteTitle.getText().toString().trim().isEmpty()){
            Toasty.makeText(this, ERROR_TITLE_EMPTY,Toasty.ERROR);
            return;
        }else  if (inpuNoteSubtitle.getText().toString().trim().isEmpty()){
            Toasty.makeText(this, ERROR_SUBTITLE_EMPTY,Toasty.ERROR);
            return;
        }else if (nemosoftsEditText.getText().toString().trim().isEmpty()){
            Toasty.makeText(this, ERROR_NOTE_EMPTY,Toasty.ERROR);
            return;
        }

        final Note note = buildNoteForSave();

        new AsyncTaskExecutor<Void, Void, Long>() {

            @Override
            protected Long doInBackground(Void unused) {
                try {
                    return dbHelper.saveNote(note);
                } catch (Exception e) {
                    return -1L;
                }
            }

              @Override
              protected void onPostExecute(Long id) {
                  if (isFinishing()){
                      return;
                  }
                  if (id != null && id > 0) {
                      String action = alreadyAvailableNote == null ? "Created" : "Edited";
                      dbHelper.addHistory(note.getId(), action, note.getTitle(), System.currentTimeMillis());
                      handleReminder(note);
                      nemosofts.notes.app.widget.NoteWidgetProvider.requestUpdate(CreateNoteActivity.this);
                  } else {
                      ReminderScheduler.cancelReminder(CreateNoteActivity.this, note.getId());
                  }
                Intent intent = new Intent();
                setResult(RESULT_OK, intent);
                finish();
            }
        }.execute();
    }

    private void initMiscellaneous() {
        linearMiscellaneous.findViewById(R.id.textMiscellaneous).setOnClickListener(view -> {
            if (bottomSheetBehavior.getState() != BottomSheetBehavior.STATE_EXPANDED){
                bottomSheetBehavior.setState(BottomSheetBehavior.STATE_EXPANDED);
            }else {
                bottomSheetBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
            }
        });

        final ImageView imageView1 = linearMiscellaneous.findViewById(R.id.imageColor1);
        final ImageView imageView2 = linearMiscellaneous.findViewById(R.id.imageColor2);
        final ImageView imageView3 = linearMiscellaneous.findViewById(R.id.imageColor3);
        final ImageView imageView4 = linearMiscellaneous.findViewById(R.id.imageColor4);
        final ImageView imageView5 = linearMiscellaneous.findViewById(R.id.imageColor5);
        final ImageView imageView6 = linearMiscellaneous.findViewById(R.id.imageColor6);
        final ImageView imageView7 = linearMiscellaneous.findViewById(R.id.imageColor7);

        linearMiscellaneous.findViewById(R.id.imageColor1).setOnClickListener(view -> {
            setectedNoteColor = COLOR_DEFAULT;
            updateColorSelection(imageView1, imageView2, imageView3, imageView4, imageView5, imageView6, imageView7);
        });

        linearMiscellaneous.findViewById(R.id.imageColor2).setOnClickListener(view -> {
            setectedNoteColor = COLOR_AMBER;
            updateColorSelection(imageView2, imageView1, imageView3, imageView4, imageView5, imageView6, imageView7);
        });

        linearMiscellaneous.findViewById(R.id.imageColor3).setOnClickListener(view -> {
            setectedNoteColor = COLOR_PURPLE;
            updateColorSelection(imageView3, imageView1, imageView2, imageView4, imageView5, imageView6, imageView7);
        });

        linearMiscellaneous.findViewById(R.id.imageColor4).setOnClickListener(view -> {
            setectedNoteColor = COLOR_BLUE;
            updateColorSelection(imageView4, imageView1, imageView2, imageView3, imageView5, imageView6, imageView7);
        });

        linearMiscellaneous.findViewById(R.id.imageColor5).setOnClickListener(view -> {
            setectedNoteColor = COLOR_PINK;
            updateColorSelection(imageView5, imageView1, imageView2, imageView3, imageView4, imageView6, imageView7);
        });

        linearMiscellaneous.findViewById(R.id.imageColor6).setOnClickListener(view -> {
            setectedNoteColor = COLOR_GREEN;
            updateColorSelection(imageView6, imageView1, imageView2, imageView3, imageView4, imageView5, imageView7);
        });

        linearMiscellaneous.findViewById(R.id.imageColor7).setOnClickListener(view -> {
            setectedNoteColor = COLOR_ORANGE;
            updateColorSelection(imageView7, imageView1, imageView2, imageView3, imageView4, imageView5, imageView6);
        });

        if (alreadyAvailableNote != null && alreadyAvailableNote.getColor() != null && !alreadyAvailableNote.getColor().trim().isEmpty()){
            switch (alreadyAvailableNote.getColor()){
                case COLOR_DEFAULT:
                    setectedNoteColor = COLOR_DEFAULT;
                    imageView1.setImageResource(R.drawable.ic_done);
                    imageView2.setImageResource(0);
                    imageView3.setImageResource(0);
                    imageView4.setImageResource(0);
                    imageView5.setImageResource(0);
                    imageView6.setImageResource(0);
                    imageView7.setImageResource(0);
                    setSubtitleIndicatorColor();
                    break;
                case COLOR_AMBER:
                    setectedNoteColor = COLOR_AMBER;
                    imageView1.setImageResource(0);
                    imageView2.setImageResource(R.drawable.ic_done);
                    imageView3.setImageResource(0);
                    imageView4.setImageResource(0);
                    imageView5.setImageResource(0);
                    imageView6.setImageResource(0);
                    imageView7.setImageResource(0);
                    setSubtitleIndicatorColor();
                    break;
                case COLOR_PURPLE:
                    setectedNoteColor = COLOR_PURPLE;
                    imageView1.setImageResource(0);
                    imageView2.setImageResource(0);
                    imageView3.setImageResource(R.drawable.ic_done);
                    imageView4.setImageResource(0);
                    imageView5.setImageResource(0);
                    imageView6.setImageResource(0);
                    imageView7.setImageResource(0);
                    setSubtitleIndicatorColor();
                    break;
                case COLOR_BLUE:
                    setectedNoteColor = COLOR_BLUE;
                    imageView1.setImageResource(0);
                    imageView2.setImageResource(0);
                    imageView3.setImageResource(0);
                    imageView4.setImageResource(R.drawable.ic_done);
                    imageView5.setImageResource(0);
                    imageView6.setImageResource(0);
                    imageView7.setImageResource(0);
                    setSubtitleIndicatorColor();
                    break;
                case COLOR_PINK:
                    setectedNoteColor = COLOR_PINK;
                    imageView1.setImageResource(0);
                    imageView2.setImageResource(0);
                    imageView3.setImageResource(0);
                    imageView4.setImageResource(0);
                    imageView5.setImageResource(R.drawable.ic_done);
                    imageView6.setImageResource(0);
                    imageView7.setImageResource(0);
                    setSubtitleIndicatorColor();
                    break;
                case COLOR_GREEN:
                    setectedNoteColor = COLOR_GREEN;
                    imageView1.setImageResource(0);
                    imageView2.setImageResource(0);
                    imageView3.setImageResource(0);
                    imageView4.setImageResource(0);
                    imageView5.setImageResource(0);
                    imageView6.setImageResource(R.drawable.ic_done);
                    imageView7.setImageResource(0);
                    setSubtitleIndicatorColor();
                    break;
                case COLOR_ORANGE:
                    setectedNoteColor = COLOR_ORANGE;
                    imageView1.setImageResource(0);
                    imageView2.setImageResource(0);
                    imageView3.setImageResource(0);
                    imageView4.setImageResource(0);
                    imageView5.setImageResource(0);
                    imageView6.setImageResource(0);
                    imageView7.setImageResource(R.drawable.ic_done);
                    setSubtitleIndicatorColor();
                    break;
                default:
                    setectedNoteColor = COLOR_DEFAULT;
                    updateColorSelection(imageView1, imageView2, imageView3, imageView4, imageView5, imageView6, imageView7);
                    break;
            }
        }

        linearMiscellaneous.findViewById(R.id.layoutAddImage).setOnClickListener(view -> {
            bottomSheetBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
            if (checkPer()){
                selectImage();
            } else {
                checkPer();
            }
        });

        linearMiscellaneous.findViewById(R.id.layoutAddUrl).setOnClickListener(view -> {
            bottomSheetBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
            showAddURLDialog();
        });

        linearMiscellaneous.findViewById(R.id.layoutAddAudio).setOnClickListener(view -> {
            bottomSheetBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
            if (checkRecordPermission()) {
                toggleRecording();
            }
        });

        linearMiscellaneous.findViewById(R.id.layoutAddReminder).setOnClickListener(view -> {
            bottomSheetBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
            if (ensureNotificationPermission()) {
                openReminderPicker();
            }
        });

        linearMiscellaneous.findViewById(R.id.layoutExport).setOnClickListener(view -> {
            bottomSheetBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
            if (checkPer()){
                showExportDialog();
            } else {
                checkPer();
            }
        });

        linearMiscellaneous.findViewById(R.id.layoutShareNote).setOnClickListener(view -> {
            bottomSheetBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
            if (inpuNoteTitle.getText().toString().trim().isEmpty()){
                Toasty.makeText(this, ERROR_TITLE_EMPTY,Toasty.ERROR);
            } else  if (inpuNoteSubtitle.getText().toString().trim().isEmpty()){
                Toasty.makeText(this, ERROR_NOTE_EMPTY,Toasty.ERROR);
            } else {
                Intent sendIntent = new Intent();
                sendIntent.setAction(Intent.ACTION_SEND);
                sendIntent.putExtra(Intent.EXTRA_TEXT,
                        inpuNoteTitle.getText().toString() + "\n\n" +
                                textDeteTime.getText().toString() + "\n\n" +
                                inpuNoteSubtitle.getText().toString() + "\n\n" +
                                Html.fromHtml(nemosoftsEditText.toHtml()) + "\n" +
                                "https://play.google.com/store/apps/details?id=" + getPackageName()
                );
                sendIntent.setType("text/plain");
                startActivity(sendIntent);
            }
        });
    }

    private void showExportDialog() {
        if (dialogExport == null) {
            AlertDialog.Builder builder = new AlertDialog.Builder(CreateNoteActivity.this);
            View view = LayoutInflater.from(this).inflate(R.layout.dialog_export_file, null);
            builder.setView(view);
            dialogExport = builder.create();
            if (dialogExport.getWindow() != null){
                dialogExport.getWindow().setBackgroundDrawable(new ColorDrawable(0));
            }
            view.findViewById(R.id.ll_export_image_file).setOnClickListener(view1 -> {
                dialogExport.dismiss();
                if (checkPer()){
                    saveImage();
                } else {
                    checkPer();
                }
            });

            view.findViewById(R.id.ll_export_txt_file).setOnClickListener(view12 -> {
                dialogExport.dismiss();
                saveResults();
            });

            view.findViewById(R.id.textCancel).setOnClickListener(view13 -> dialogExport.dismiss());
        }

        dialogExport.show();
    }

    private void saveImage() {
        Bitmap bitmap;
        bitmap= viewToBitmap(llScroll);
        saveImageToExternalStorage(bitmap);
    }

    public Bitmap viewToBitmap(View view) {
        Bitmap bitmap = Bitmap.createBitmap(view.getWidth(), view.getHeight(), Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);
        view.draw(canvas);
        return bitmap;
    }

    private void saveImageToExternalStorage(Bitmap finalBitmap) {
        String iconsStoragePath = Environment.getExternalStorageDirectory().toString() + File.separator + Environment.DIRECTORY_DOCUMENTS + File.separator + getString(R.string.app_name);
        File sdIconStorageDir = new File(iconsStoragePath);
        if (!sdIconStorageDir.exists() && !sdIconStorageDir.mkdirs()) {
            Log.e(TAG, "Unable to create image export directory at " + iconsStoragePath);
            Toast.makeText(this, R.string.err_cannot_use_features, Toast.LENGTH_SHORT).show();
            return;
        }
        generator = generatorNumber();
        String fname = "Image_" + generator + ".jpg";
        File file = new File(sdIconStorageDir, fname);

        try (FileOutputStream out = new FileOutputStream(file)) {
            finalBitmap.compress(Bitmap.CompressFormat.JPEG, 100, out);
            out.flush();
            Toast.makeText(this, "Image is created!!!", Toast.LENGTH_SHORT).show();
        } catch (Exception e) {
            Log.e(TAG, "Failed to save image", e);
            Toast.makeText(this, R.string.err_cannot_use_features, Toast.LENGTH_LONG).show();
        }

        // Tell the media scanner about the new file so that it is
        // immediately available to the user.
        MediaScannerConnection.scanFile(this, new String[] { file.toString() }, null, (path, uri) -> {
            Log.i("ExternalStorage", "Scanned " + path + ":");
            Log.i("ExternalStorage", "-> uri=" + uri);
        });

        openGeneratedJPG();
    }

    private void openGeneratedJPG(){
        String iconsStoragePath = Environment.getExternalStorageDirectory().toString() + File.separator + Environment.DIRECTORY_DOCUMENTS + File.separator + getString(R.string.app_name);
        File sdIconStorageDir = new File(iconsStoragePath);
        if (!sdIconStorageDir.exists() && !sdIconStorageDir.mkdirs()) {
            Log.e(TAG, "Unable to create image directory for opening");
            return;
        }

        if (sdIconStorageDir.exists()) {
            String fname = "Image_" + generator + ".jpg";
            File file = new File(sdIconStorageDir, fname);
            Intent intent=new Intent(Intent.ACTION_VIEW);
            Uri uri = Uri.fromFile(file);
            intent.setDataAndType(uri, "image/*");
            intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
            try {
                startActivity(intent);
            } catch(ActivityNotFoundException e) {
                Toast.makeText(CreateNoteActivity.this, "No Application available to view JPG", Toast.LENGTH_LONG).show();
            }
        }
    }

    private int generatorNumber() {
        return RANDOM.nextInt(10000);
    }

    private void showDeletNoteDialog() {
        DialogUtil.trashRecycleDialog(this, new DialogUtil.DeleteListener() {
            @Override
            public void onDelete() {
                new AsyncTaskExecutor<String, String, String>() {

                    @Override
                    protected void onPreExecute() {
                        super.onPreExecute();
                        deleteSaveNote();
                    }

                    @Override
                    protected String doInBackground(String strings) {
                        try {
                            dbHelper.deleteNote(alreadyAvailableNote.getId());
                            return "1";
                        } catch (Exception e) {
                            return "0";
                        }
                    }

                    @SuppressLint("NotifyDataSetChanged")
                    @Override
                    protected void onPostExecute(String s) {
                        if (isFinishing()){
                            return;
                        }
                        Callback.setRecreate(true);
                        finish();
                    }
                }.execute();
            }

            @Override
            public void onCancel() {
                // this method is empty
            }
        });
    }

    private void updateReminderUI() {
        if (reminderTime <= 0) {
            clearReminder();
            return;
        }
        layoutReminder.setVisibility(View.VISIBLE);
        String formatted = new SimpleDateFormat("EEE, dd MMM yyyy hh:mm a", Locale.getDefault())
                .format(new Date(reminderTime));
        textReminder.setText(getString(R.string.reminder_at, formatted));
    }

    private void clearReminder() {
        reminderTime = 0L;
        if (layoutReminder != null) {
            layoutReminder.setVisibility(View.GONE);
        }
        if (textReminder != null) {
            textReminder.setText(null);
        }
    }

    private void showAudio(long durationMs) {
        layoutAudio.setVisibility(View.VISIBLE);
        textAudioDuration.setText(formatDuration(durationMs));
        buttonAudioAction.setImageResource(isPlaying ? R.drawable.ic_pause : R.drawable.ic_play_arrow);
    }

    private void clearAudio() {
        stopPlayback();
        stopRecordingSilently();
        if (audioPath != null && !audioPath.isEmpty()) {
            try {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    if (!Files.deleteIfExists(Paths.get(audioPath))) {
                        Log.w(TAG, "Unable to delete audio file " + audioPath);
                    }
                } else {
                    if (!new File(audioPath).delete()) {
                        Log.w(TAG, "Unable to delete audio file " + audioPath);
                    }
                }
            } catch (Exception e) {
                Log.e(TAG, "Failed to delete audio file " + audioPath, e);
            }
        }
        audioPath = "";
        audioDuration = 0L;
        layoutAudio.setVisibility(View.GONE);
        textAudioDuration.setText(null);
    }

    private void updateColorSelection(ImageView selected, ImageView... others) {
        selected.setImageResource(R.drawable.ic_done);
        for (ImageView other : others) {
            other.setImageResource(0);
        }
        setSubtitleIndicatorColor();
    }

    private boolean ensureNotificationPermission() {
        if (Build.VERSION.SDK_INT >= 33 && ContextCompat.checkSelfPermission(this, Manifest.permission.POST_NOTIFICATIONS) != PackageManager.PERMISSION_GRANTED) {
            ActivityCompat.requestPermissions(this, new String[]{Manifest.permission.POST_NOTIFICATIONS}, REQ_POST_NOTIFICATIONS);
            Toasty.makeText(this, getString(R.string.notification), Toasty.INFO);
            return false;
        }
        return true;
    }

    private boolean checkRecordPermission() {
        if (ContextCompat.checkSelfPermission(this, Manifest.permission.RECORD_AUDIO) != PackageManager.PERMISSION_GRANTED) {
            ActivityCompat.requestPermissions(this, new String[]{Manifest.permission.RECORD_AUDIO}, REQ_RECORD_AUDIO);
            Toasty.makeText(this, getString(R.string.audio_permission_denied), Toasty.INFO);
            return false;
        }
        return true;
    }

    private void toggleRecording() {
        if (isRecording) {
            stopRecording();
        } else {
            startRecording();
        }
    }

    private void startRecording() {
        try {
            clearAudio();
            File dir = new File(getFilesDir(), "notes_audio");
            if (!dir.exists() && !dir.mkdirs()) {
                Log.e(TAG, "Unable to create audio directory");
                Toasty.makeText(this, getString(R.string.err_cannot_use_features), Toasty.ERROR);
                return;
            }
            audioPath = new File(dir, "audio_" + System.currentTimeMillis() + ".m4a").getAbsolutePath();
            mediaRecorder = new MediaRecorder();
            mediaRecorder.setAudioSource(MediaRecorder.AudioSource.MIC);
            mediaRecorder.setOutputFormat(MediaRecorder.OutputFormat.MPEG_4);
            mediaRecorder.setAudioEncoder(MediaRecorder.AudioEncoder.AAC);
            mediaRecorder.setOutputFile(audioPath);
            mediaRecorder.prepare();
            mediaRecorder.start();
            isRecording = true;
            recordStartTime = System.currentTimeMillis();
            startRecordingTimer();
            layoutAudio.setVisibility(View.VISIBLE);
            buttonAudioAction.setImageResource(R.drawable.ic_pause);
            Toasty.makeText(this, getString(R.string.audio_recording), Toasty.INFO);
        } catch (Exception e) {
            isRecording = false;
            audioPath = "";
            Toasty.makeText(this, e.getMessage(), Toasty.ERROR);
        }
    }

    private void stopRecording() {
        try {
            if (mediaRecorder != null) {
                mediaRecorder.stop();
                mediaRecorder.release();
                mediaRecorder = null;
            }
            isRecording = false;
            audioDuration = System.currentTimeMillis() - recordStartTime;
            if (audioDuration < 500) {
                clearAudio();
                Toasty.makeText(this, getString(R.string.reminder_time_invalid), Toasty.ERROR);
                return;
            }
            stopRecordingTimer();
            showAudio(audioDuration);
        } catch (Exception e) {
            isRecording = false;
            Toasty.makeText(this, e.getMessage(), Toasty.ERROR);
        }
    }

    private void stopRecordingSilently() {
        try {
            if (mediaRecorder != null) {
                mediaRecorder.stop();
                mediaRecorder.release();
                mediaRecorder = null;
            }
        } catch (Exception e) {
            Log.w(TAG, "Silently stopping recorder failed", e);
        }
        isRecording = false;
        stopRecordingTimer();
    }

    private void togglePlay() {
        if (audioPath == null || audioPath.trim().isEmpty()) {
            Toasty.makeText(this, getString(R.string.audio), Toasty.INFO);
            return;
        }
        if (isRecording) {
            stopRecording();
            return;
        }
        if (isPlaying) {
            stopPlayback();
        } else {
            startPlayback();
        }
    }

    private void startPlayback() {
        try {
            stopPlayback();
            mediaPlayer = new MediaPlayer();
            mediaPlayer.setDataSource(audioPath);
            mediaPlayer.setOnCompletionListener(mp -> {
                isPlaying = false;
                buttonAudioAction.setImageResource(R.drawable.ic_play_arrow);
            });
            mediaPlayer.prepare();
            mediaPlayer.start();
            isPlaying = true;
            buttonAudioAction.setImageResource(R.drawable.ic_pause);
        } catch (Exception e) {
            isPlaying = false;
            Toasty.makeText(this, e.getMessage(), Toasty.ERROR);
        }
    }

    private void stopPlayback() {
        try {
            if (mediaPlayer != null) {
                mediaPlayer.stop();
                mediaPlayer.release();
                mediaPlayer = null;
            }
        } catch (Exception e) {
            Log.w(TAG, "Failed to stop playback", e);
        }
        isPlaying = false;
        if (buttonAudioAction != null) {
            buttonAudioAction.setImageResource(R.drawable.ic_play_arrow);
        }
    }

    @Override
    protected void onPause() {
        super.onPause();
        stopPlayback();
        stopRecordingSilently();
    }

    private void startRecordingTimer() {
        stopRecordingTimer();
        recordingTicker = new Runnable() {
            @Override
            public void run() {
                long elapsed = System.currentTimeMillis() - recordStartTime;
                textAudioDuration.setText(getString(R.string.audio_recording) + " " + formatDuration(elapsed));
                recordingHandler.postDelayed(this, 500);
            }
        };
        recordingHandler.post(recordingTicker);
    }

    private void stopRecordingTimer() {
        if (recordingTicker != null) {
            recordingHandler.removeCallbacks(recordingTicker);
            recordingTicker = null;
        }
    }

    private String formatDuration(long durationMs) {
        long seconds = durationMs / 1000;
        long minutes = seconds / 60;
        long remSec = seconds % 60;
        return String.format(Locale.getDefault(), "%02d:%02d", minutes, remSec);
    }

    private void handleReminder(@NonNull Note note) {
        if (Build.VERSION.SDK_INT >= 33
                && ContextCompat.checkSelfPermission(this, Manifest.permission.POST_NOTIFICATIONS) != PackageManager.PERMISSION_GRANTED) {
            return;
        }
        if (note.isReminderEnabled() && note.getReminderTime() > 0 && note.getId() > 0) {
            ReminderScheduler.scheduleReminder(this, note);
        } else {
            ReminderScheduler.cancelReminder(this, note.getId());
        }
    }

    private void setSubtitleIndicatorColor(){
        GradientDrawable gradientDrawable = (GradientDrawable) viewSubtitleIndicator.getBackground();
        if (setectedNoteColor.equals(COLOR_DEFAULT)) {
            if (new ThemeEngine(this).getIsDark()) {
                gradientDrawable.setColor(Color.parseColor("#ECECEC"));
            } else {
                gradientDrawable.setColor(Color.parseColor("#121212"));
            }
        } else {
            gradientDrawable.setColor(Color.parseColor(setectedNoteColor));
        }
    }

    private void selectImage() {
        imagePickerLauncher.launch("image/*");
    }

    @NonNull
    private Boolean checkPer() {
        if (Build.VERSION.SDK_INT >= 33){
            if ((ContextCompat.checkSelfPermission(CreateNoteActivity.this, Manifest.permission.READ_MEDIA_IMAGES) != PackageManager.PERMISSION_GRANTED)) {
                requestPermissions(new String[]{Manifest.permission.READ_MEDIA_IMAGES}, PERMISSION_REQ_WRITE_EXTERNAL_STORAGE);
                return false;
            } else {
                return true;
            }
        } else if (Build.VERSION.SDK_INT >= 29) {
            if ((ContextCompat.checkSelfPermission(CreateNoteActivity.this, Manifest.permission.READ_EXTERNAL_STORAGE) != PackageManager.PERMISSION_GRANTED)) {
                requestPermissions(new String[]{Manifest.permission.READ_EXTERNAL_STORAGE}, PERMISSION_REQ_WRITE_EXTERNAL_STORAGE);
                return false;
            } else {
                return true;
            }
        } else {
            if ((ContextCompat.checkSelfPermission(CreateNoteActivity.this, Manifest.permission.WRITE_EXTERNAL_STORAGE) != PackageManager.PERMISSION_GRANTED)) {
                requestPermissions(new String[]{Manifest.permission.WRITE_EXTERNAL_STORAGE}, PERMISSION_REQ_WRITE_EXTERNAL_STORAGE);
                return false;
            }
            return true;
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        boolean canUseExternalStorage = false;
        if (requestCode == PERMISSION_REQ_WRITE_EXTERNAL_STORAGE) {
            if (grantResults.length > 0
                    && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                canUseExternalStorage = true;
            }

            if (!canUseExternalStorage) {
                Toast.makeText(CreateNoteActivity.this, getResources().getString(R.string.err_cannot_use_features), Toast.LENGTH_SHORT).show();
            }
        }

        if (requestCode == REQ_POST_NOTIFICATIONS) {
            if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                Toasty.makeText(this, getString(R.string.notification), Toasty.SUCCESS);
            } else {
                Toasty.makeText(this, getString(R.string.to_receive_notification), Toasty.INFO);
            }
        }
        if (requestCode == REQ_RECORD_AUDIO) {
            if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                Toasty.makeText(this, getString(R.string.audio), Toasty.SUCCESS);
                toggleRecording();
            } else {
                Toasty.makeText(this, getString(R.string.audio_permission_denied), Toasty.ERROR);
            }
        }
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
    }

    private void handleImagePicked(@Nullable Uri selectedImageUri) {
        if (selectedImageUri == null) {
            return;
        }
        try (InputStream inputStream = getContentResolver().openInputStream(selectedImageUri)) {
            Bitmap bitmap = BitmapFactory.decodeStream(inputStream);
            imageNote.setImageBitmap(bitmap);
            imageNote.setVisibility(View.VISIBLE);
            findViewById(R.id.create_imageRemoveImage).setVisibility(View.VISIBLE);
            String copiedPath = copyImageToInternalStorage(selectedImageUri);
            if (copiedPath == null) {
                String localPath = getPathFromUri(selectedImageUri);
                setectedImagePath = localPath != null ? localPath : selectedImageUri.toString();
            } else {
                setectedImagePath = copiedPath;
            }
            loadNoteImage(setectedImagePath);
        } catch (Exception e) {
            Log.e(TAG, "Failed to handle picked image", e);
            Toast.makeText(this, e.getMessage(), Toast.LENGTH_SHORT).show();
        }
    }

    @SuppressLint("Range")
    public String getPathFromUri(Uri uri) {
        return MediaPath.getPathImage(this, uri);
    }

    private String copyImageToInternalStorage(Uri uri) {
        try (InputStream inputStream = getContentResolver().openInputStream(uri)) {
            if (inputStream == null) return null;
            File dir = new File(getFilesDir(), "notes_images");
            if (!dir.exists() && !dir.mkdirs()) {
                Log.e(TAG, "Unable to create internal image directory");
                return null;
            }
            String fileName = "note_img_" + System.currentTimeMillis() + ".jpg";
            File outFile = new File(dir, fileName);
            try (FileOutputStream outputStream = new FileOutputStream(outFile)) {
                byte[] buffer = new byte[4096];
                int bytesRead;
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, bytesRead);
                }
                outputStream.flush();
                return outFile.getAbsolutePath();
            }
        } catch (Exception e) {
            Log.e(TAG, "Failed to copy image to internal storage", e);
            return null;
        }
    }

    private void loadNoteImage(String path) {
        try {
            if (path == null || path.trim().isEmpty()) {
                return;
            }
            if (path.startsWith("content://")) {
                Uri uri = Uri.parse(path);
                try (InputStream inputStream = getContentResolver().openInputStream(uri)) {
                    Bitmap bitmap = BitmapFactory.decodeStream(inputStream);
                    imageNote.setImageBitmap(bitmap);
                }
            } else {
                imageNote.setImageBitmap(BitmapFactory.decodeFile(path));
            }
            imageNote.setVisibility(View.VISIBLE);
            findViewById(R.id.create_imageRemoveImage).setVisibility(View.VISIBLE);
        } catch (Exception e) {
            Log.e(TAG, "Failed to load note image", e);
        }
    }


    private void showAddURLDialog(){
        DialogUtil.addURLDialog(this, string -> {
            if (string.isEmpty()){
                Toast.makeText(CreateNoteActivity.this,"Enter URL", Toast.LENGTH_SHORT).show();
                return;
            }
            textWebURL.setText(string);
            layoutWebURL.setVisibility(View.VISIBLE);
        });
    }

    private void deleteSaveNote() {
        if (inpuNoteTitle.getText().toString().trim().isEmpty()){
            Toasty.makeText(this, ERROR_TITLE_EMPTY,Toasty.ERROR);
            return;
        } else if (inpuNoteSubtitle.getText().toString().trim().isEmpty()){
            Toasty.makeText(this, ERROR_SUBTITLE_EMPTY,Toasty.ERROR);
            return;
        } else if (nemosoftsEditText.getText().toString().trim().isEmpty()){
            Toasty.makeText(this, ERROR_NOTE_EMPTY,Toasty.ERROR);
            return;
        }

        final Note note = buildNoteForSave();

        new AsyncTaskExecutor<Void, Void, Long>() {

            @Override
            protected Long doInBackground(Void unused) {
                try {
                    dbHelper.addDeletedNote(note);
                    return (long) note.getId();
                } catch (Exception e) {
                    return -1L;
                }
            }

            @Override
            protected void onPostExecute(Long aLong) {
                ReminderScheduler.cancelReminder(CreateNoteActivity.this, note.getId());
            }
        }.execute();
    }

    private void saveNoteOnBackPressed() {
        final Note note = buildNoteForSave();

        new AsyncTaskExecutor<Void, Void, Long>() {

            @Override
            protected Long doInBackground(Void unused) {
                try {
                    return dbHelper.saveNote(note);
                } catch (Exception e) {
                    return -1L;
                }
            }

              @Override
              protected void onPostExecute(Long id) {
                  if (isFinishing()){
                      return;
                  }
                  if (id != null && id > 0) {
                      String action = alreadyAvailableNote == null ? "Created" : "Edited";
                      dbHelper.addHistory(note.getId(), action, note.getTitle(), System.currentTimeMillis());
                      handleReminder(note);
                      nemosofts.notes.app.widget.NoteWidgetProvider.requestUpdate(CreateNoteActivity.this);
                  } else {
                      ReminderScheduler.cancelReminder(CreateNoteActivity.this, note.getId());
                  }
                Intent intent = new Intent();
                setResult(RESULT_OK, intent);
                finish();
            }
        }.execute();
    }

    public void saveResults() {
        String textNote = " ";
        String textTitle = " ";
        String textSubtitle =" ";

        if (!nemosoftsEditText.getText().toString().isEmpty()){
            textNote = nemosoftsEditText.getText().toString();
        }
        if (!inpuNoteTitle.getText().toString().isEmpty()){
            textTitle = inpuNoteTitle.getText().toString();
        }
        if (!inpuNoteSubtitle.getText().toString().isEmpty()){
            textSubtitle = inpuNoteSubtitle.getText().toString();
        }

        String text = textTitle+ "\n"
                +"\n"
                +textSubtitle +"\n"
                +"\n"
                +textNote;

        String iconsStoragePath = Environment.getExternalStorageDirectory().toString() + File.separator + Environment.DIRECTORY_DOCUMENTS + File.separator + getString(R.string.app_name);
        File sdIconStorageDir = new File(iconsStoragePath);
        if (!sdIconStorageDir.exists() && !sdIconStorageDir.mkdirs()) {
            Log.e(TAG, "Unable to create export directory at " + iconsStoragePath);
            Toast.makeText(this, R.string.err_cannot_use_features, Toast.LENGTH_SHORT).show();
            return;
        }
        generator = generatorNumber();
        String fname = "notes_" + generator + ".txt";
        File file = new File(sdIconStorageDir, fname);

        try (FileOutputStream out = new FileOutputStream(file)) {
            out.write(text.getBytes());
            nemosoftsEditText.getText().clear();
            out.flush();
            Toast.makeText(this, "txt is created!!!", Toast.LENGTH_SHORT).show();
        } catch (Exception e) {
            Log.e(TAG, "Failed to save txt export", e);
            Toast.makeText(this, R.string.err_cannot_use_features, Toast.LENGTH_LONG).show();
        }
    }
}